<?php
class FieldManager {
    private $db;
    private $allowedTypes = [
        'text' => 'VARCHAR(255)',
        'longtext' => 'LONGTEXT',
        'number' => 'INT'
    ];

    public function __construct($db) {
        $this->db = $db;
    }

    public function addField($table, $name, $type) {
        // Validate
        if (!preg_match('/^[a-z0-9_]+$/', $name)) {
            throw new Exception("Field name can only contain a-z, 0-9, and underscores");
        }

        if (!isset($this->allowedTypes[$type])) {
            throw new Exception("Invalid field type");
        }

        // Safely alter table
        $this->db->query(
            "ALTER TABLE `$table` ADD COLUMN `$name` {$this->allowedTypes[$type]}"
        );
    }

    public function validateFieldValue($type, $value) {
        if ($value === '') return null;

        switch ($type) {
            case 'number':
                return is_numeric($value) ? (int)$value : null;
            case 'text':
                return substr(trim($value), 0, 255);
            default:
                return $value;
        }
    }

    public function deleteField($table, $field) {
        $this->db->query("ALTER TABLE `$table` DROP COLUMN `$field`");
    }

    public function changeFieldType($table, $field, $newType) {
        $this->db->query("ALTER TABLE `$table` MODIFY COLUMN `$field` $newType");
    }
}